#ifndef __CRegistry__
#define __CRegistry__


#include "../Basics/CSystemString.hpp"
#include "../Collections/TCountedPointerArray.hpp"
#include "CRegistryInfo.hpp"

//	===========================================================================

using Exponent::Basics::CSystemString;
using Exponent::Basics::CString;
using Exponent::Collections::TCountedPointerArray;
using Exponent::Host::CRegistryInfo;

//	===========================================================================

namespace Exponent
{
	namespace Host
	{
		/**
		 * @class CRegistry CRegistry.hpp
		 * @brief Read / Write to the registry
		 *
		 * @date 21/08/2004
		 * @author Paul Chana
		 * @version 1.0.0 Initial version
		 *
		 * @note All contents of this source code are copyright 2005 Exp Digital Uk.\n
		 * This source file is covered by the licence conditions of the Infinity API. You should have recieved a copy\n
		 * with the source code. If you didnt, please refer to http://www.expdigital.co.uk
		 * All content is the Intellectual property of Exp Digital Uk.\n
		 * Certain sections of this code may come from other sources. They are credited where applicable.\n
		 * If you have comments, suggestions or bug reports please visit http://support.expdigital.co.uk
		 *
		 * $Id: CRegistry.hpp,v 1.4 2007/02/08 21:06:44 paul Exp $
		 */
		class CRegistry
		{
		public:

//	===========================================================================

			/**
			 * @enum ERegistry
			 * @brief Registry enumeration, check specifics for system
			 */
			enum ERegistry
			{
				e_localMachine = 0,		/**< Invalid on MACOSX (WIN32 only), if used on macosx will be replaced by current user */
				e_currentUser,			/**< Valid on MACOSX */
				e_currentApplication,	/**< Invalid on WIN32 (MACOSX only), if used on windows will cause an error */
			};

//	===========================================================================

			/**
			 * Read from the registry
			 * @param registry The registry to delete from
			 * @param registryKey the registryKey (eg Exponent\\GFlux)
			 * @param key the key to read (eg DataPath)
			 * @param input On return is filled with the registry value
			 * @retval bool True if read properly, false otherwise
			 */
			static bool readFromRegistry(const ERegistry registry, const CString &registryKey, const CString &key, CSystemString &input);

			/**
			 * Write to the registry
			 * @param registry The registry to delete from
			 * @param registryKey the registryKey (eg Exponent\\GFlux)
			 * @param key the key to write (eg DataPath)
			 * @param value The value to give the key
			 * @retval bool True if written properly, false otherwise
			 */
			static bool writeToRegistry(const ERegistry registry, const CString &registryKey, const CString &key, const CString &value);

			/**
			 * Add to the registry.. WARNING! WARNING! WARNING! Could potentially be very damaging to system
			 * @param registry The registry to add to
			 * @param registryKey the registryKey (eg Exponent\\GFlux)
			 * @retval bool True if added properly, false otherwise
			 */
			static bool createRegistryEntry(const ERegistry registry, const CString &registryKey);

			/**
			 * Delete from the registry.. WARNING! WARNING! WARNING! Could potentially be very damaging to system
			 * @param registry The registry to delete from
			 * @param registryKey the registryKey (eg Exponent\\GFlux)
			 * @param key the key to delete (eg DataPath)
			 * @retval bool True if deleted properly, false otherwise
			 */
			static bool deleteRegistryEntry(const ERegistry registry, const CString &registryKey, const CString &key);

			/**
			 * Enumerate a registry key
			 * @param registry The registry to enumerate
			 * @param registryKey The root key
			 * @retval TCountedPointerArray<CRegistryInfo> The array of CRegistryInfo
			 * @see CRegistryInfo
			 */
			static TCountedPointerArray<CRegistryInfo> enumerateRegistryKeyValues(const ERegistry registry, const CString &registryKey);

//	===========================================================================

		};
	}
}
#endif	// End of CRegistry.hpp
